from sqlalchemy.exc import StatementError
from sqlalchemy import join, Table, MetaData, select, desc, cast, Numeric, or_, and_, union_all, asc, func
import qrcode
# from sqlalchemy import *
# from sqlalchemy.orm import *
# from sqlalchemy.ext.declarative import declarative_base
from PIL import Image
from re import sub, match
import webbrowser
import os
import plotly.io as pio
import plotly.graph_objects as px
from flask import Flask, render_template, request, redirect, url_for, jsonify
from flask_sqlalchemy import SQLAlchemy
from sqlalchemy import text, inspect
import pandas as pd
import json
import colorama
import logging
import sys
import importlib
from tqdm import tqdm

app = Flask(__name__, template_folder=os.path.dirname(os.path.abspath(__file__)))


class TableParams:
    def __init__(self, find_conditions: dict):
        with (app.app_context()):
            if 'имя' not in find_conditions.keys():
                name = sub(r'[.:\s-]', '_', find_conditions['показатель'])
            else:
                name = find_conditions['имя']
            if name in queries.keys():
                counter = 1
                while name + f"_№{counter}" in queries.keys():
                    counter += 1
                name += f"_№{counter}"
            self.find_conditions = find_conditions
            queries[name] = self

    def find(self):
        with app.app_context():
            for tableName in inspect(db.engine).get_table_names():
                connection = db.engine.connect()
                metadata = MetaData()
                find_table = Table(tableName, metadata, autoload_with=db.engine)
                cur_find_conditions = self.find_conditions.copy()
                other_conditions = self.find_conditions.copy()
                dates = cur_find_conditions["дата"]
                del other_conditions['дата']
                if 'муниципальное_образование' in cur_find_conditions.keys():
                    del other_conditions['муниципальное_образование']
                del other_conditions['показатель']
                if 'имя' in cur_find_conditions.keys():
                    del other_conditions['имя']
                sort_table = Table("ПОКАЗАТЕЛЬ", metadata, autoload_with=db.engine)

                base_query = select(
                    find_table.c.муниципальное_образование,
                    find_table.c.показатель,
                    find_table.c.рейтинг,
                    sort_table.c.сортировка_показатель,
                    sort_table.c.фильтр
                ).where(
                    and_(
                        find_table.c.дата == dates[0],
                        find_table.c.показатель == sort_table.c.сортировка_показатель,
                        find_table.c.показатель == cur_find_conditions['показатель']
                        if cur_find_conditions['показатель'] is not None else find_table.c.показатель.like("%"),
                        find_table.c.муниципальное_образование.in_(
                            cur_find_conditions["муниципальное_образование"]) if
                        cur_find_conditions[
                            "муниципальное_образование"] is not None else find_table.c.муниципальное_образование.like(
                            "%"),
                        *[getattr(find_table.c, key) == other_conditions[key] for key in other_conditions.keys()]
                    )
                ).distinct()
                # print('↓'*100,'\n',str(base_query.compile(db.engine)), '\n', '↑'*100)

                join_query = select(
                    base_query.c.муниципальное_образование,
                    base_query.c.показатель,
                    base_query.c.рейтинг,
                    base_query.c.сортировка_показатель,
                    base_query.c.фильтр
                ).join(
                    find_table,
                    onclause=and_(
                        base_query.c.муниципальное_образование == find_table.c.муниципальное_образование,
                        base_query.c.показатель == find_table.c.показатель,
                        find_table.c.дата == dates[1]
                    )
                ).where(
                    and_(
                        find_table.c.показатель == cur_find_conditions['показатель']
                        if cur_find_conditions['показатель'] is not None else find_table.c.показатель.like("%"),
                        find_table.c.муниципальное_образование.in_(
                            cur_find_conditions["муниципальное_образование"]) if
                        cur_find_conditions[
                            "муниципальное_образование"] is not None else find_table.c.муниципальное_образование.like(
                            "%"),
                        *[getattr(find_table.c, key) == other_conditions[key] for key in other_conditions.keys()]
                    )
                ).distinct()

                for date in dates:
                    date_alias = find_table.alias(name=f'{sub(r"[ /().:-]", "_", date)}')
                    join_query = join_query.outerjoin(
                        date_alias,
                        onclause=and_(
                            base_query.c.муниципальное_образование == date_alias.c.муниципальное_образование,
                            base_query.c.показатель == date_alias.c.показатель,
                            date_alias.c.дата == date
                        )
                    )
                    join_query = join_query.add_columns(
                        date_alias.c.значение.label(sub(r'[ /().:-]', '_', date))
                    )

                if self.find_conditions['показатель'] is None:
                    join_query = join_query.order_by(cast(sort_table.c.фильтр, Numeric))
                else:
                    join_query = join_query.order_by(cast(base_query.c.рейтинг, Numeric).desc())
                filtered_columns = [column for column in join_query.selected_columns if
                                    column.name not in ["фильтр", "сортировка_показатель"]]
                join_query = join_query.with_only_columns(*filtered_columns)
                # print(str(join_query.compile(db.engine)))
                return connection.execute(join_query)


def get_boolean_config(config_module=""):
    """
    Извлекает все булевы параметры из модуля конфигурации.

    Args:
        config_module: Модуль конфигурации, загруженный функцией load_config().

    Returns:
        Словарь, где ключи — имена булевых параметров, а значения — их значения (bool).
        Возвращает None, если config_module равен None.
    """
    if config_module == "":
        global config
        config_module = config
    if config_module is None:
        return None
    boolean_config = {}
    for name, value in vars(config_module).items():
        if isinstance(value, bool):
            boolean_config[name] = value
    return boolean_config


def load_config():
    possible_paths = [folder_path]
    for path in possible_paths:
        config_path = f"{path}/config.py"
        if os.path.exists(config_path):
            try:
                spec = importlib.util.spec_from_file_location("config", config_path)
                config_module = importlib.util.module_from_spec(spec)
                spec.loader.exec_module(config_module)
                sys.modules["config"] = config_module
                return config_module
            except FileNotFoundError:
                pass
    raise FileNotFoundError


def generate_and_save_qr(link, filename="qr_code"):
    """
    Генерирует QR-код из ссылки и сохраняет его в файл PNG размера 200x200 в папку /static.

    Args:
        link (str): Ссылка для кодирования в QR-код.
        filename (str, optional): Имя файла без расширения. По умолчанию "qr_code".
    """
    if config.DEVELOPER_MODE:
        static_dir = "static"
    else:
        static_dir = f"{config.DATA_PATH}\\static".replace("\\\\", "\\")
    if not os.path.exists(static_dir):
        os.makedirs(static_dir)  # создаем папку /static, если она не существует
    filepath = os.path.join(static_dir, sub(r'''[ /()"'.:-]''', '_', filename) + ".png")
    global caption_filename
    caption_filename[filename] = sub(r'''[ /()"'.:-]''', '_', filename) + ".png"
    qr = qrcode.QRCode(
        version=1,
        error_correction=qrcode.constants.ERROR_CORRECT_L,
        box_size=10,  # Выберем box_size, чтобы размер изображения был близок к 200x200
        border=4,
    )
    qr.add_data(link)
    qr.make(fit=True)

    img = qr.make_image(fill_color="black", back_color="white").convert("RGB")

    # Рассчитаем размер картинки, сохраняя пропорции, не больше 200x200
    img_width, img_height = img.size
    if img_width > 200 or img_height > 200:
        if img_width > img_height:
            new_width = 200
            new_height = int(200 * (img_height / img_width))
        else:
            new_height = 200
            new_width = int(200 * (img_width / img_height))

        img = img.resize((new_width, new_height), Image.Resampling.LANCZOS)
    # Создаем белую область 200x200 и вставляем qr-код по центру
    final_image = Image.new("RGB", (200, 200), "white")

    x_offset = (200 - img.width) // 2
    y_offset = (200 - img.height) // 2

    final_image.paste(img, (x_offset, y_offset))
    final_image.save(filepath)


def update_config_file(key, value):
    """Обновляет переменную в файле config.py"""
    with open("config.py", "r") as f:
        lines = f.readlines()

    with open("config.py", "w") as f:
        for line in lines:
            if line.startswith(f"{key}"):
                f.write(f"{key} = {value}\n")
            else:
                f.write(line)
        f.close()


@app.route('/update_setting', methods=['POST'])
def update_setting():
    global config
    answer = request.form
    for key, value in answer.items():
        setattr(config, key, value == 'True')
        update_config_file(key, value)
    # return redirect(url_for(f'table', key='Главная страница'))
    return redirect(request.referrer)


@app.route('/table/<string:key>', methods=['POST', 'GET'])
def table(key):
    table_query = queries[key]

    histogram, histogram_rate = None, None
    schema = None
    pairs = None
    speed = None
    if key == "Глоссарий":
        return render_template('index.html',
                               active_tab="Глоссарий",
                               tabs=get_menu_list(),
                               settings=get_boolean_config(),
                               caption_filename=caption_filename,
                               caption_link=caption_link)
    if key == "Главная страница":
        temp = get_menu_list().copy()
        del temp['Главная страница']
        return render_template('index.html',
                               active_tab=None,
                               tabs=temp,
                               caption_filename=caption_filename,
                               caption_link=caption_link)
    if key not in queries.keys():
        return render_template('index.html',
                               alertText="НЕИЗВЕСТНАЯ ТАБЛИЦА",
                               tabs=get_menu_list()
                               )
    selected_dates = [date for date in request.form.getlist('дата')]
    selected_regs = [reg for reg in request.form.getlist('муниципальное_образование')]
    if request.method == "POST":
        if len(selected_dates) != 2 or (len(selected_regs)>1 and table_query.find_conditions['показатель'] is None):
            return render_template('index.html',
                                   alertText="ВЫБРАНО НЕВЕРНОЕ КОЛ-ВО ПОКАЗАТЕЛЕЙ",
                                   tabs=get_menu_list()
                                   )
        table_query.find_conditions["дата"] = selected_dates
        if len(selected_regs) == 0:
            table_query.find_conditions["муниципальное_образование"] = None
        else:
            table_query.find_conditions["муниципальное_образование"] = selected_regs
    if isinstance(table_query, str):  # строчный или текстовый запрос
        res = db.session.execute(text(table_query))
    else:
        res = table_query.find()
    axis_signatures, rows, dashboard_data = res_to_data(res)
    if not isinstance(table_query, str) and len(
            set(list(dashboard_data['показатель']))) == 1:  # строить или не строить диаграммы
        histogram = generate_histogram(rows, list(dashboard_data.keys()), axis_signatures)
        schema = table_query.find_conditions
        pairs = None
        res = table_query.find()
        axis_signatures, rows, dashboard_data = res_to_data(res)
        histogram_rate = generate_histogram(rows, list(dashboard_data.keys()), axis_signatures, type='рейтинг')
    elif len(set(list(dashboard_data['показатель']))) > 1:
        pairs = dashboard_data_to_pairs(dashboard_data)
        speed = float(pairs['Индекс ФДˏ балл (из 100 возможных)'][3])
        del pairs['Индекс ФДˏ балл (из 100 возможных)']
        histogram, histogram_rate = None, None
        schema = table_query.find_conditions
    return render_template('index.html',
                           active_tab="Кастомный запрос",
                           table=dashboard_data,
                           tabs=get_menu_list(),
                           settings=get_boolean_config(),
                           plot_data_main=histogram,
                           plot_data_rate=histogram_rate,
                           filter_params=get_list_of_params_for_query(schema),
                           pairs=pairs,
                           speed=speed
                           )


def dashboard_data_to_pairs(data):
    result = {}
    indicators = data.get('показатель')

    if not indicators:
        return result

    for i, indicator in enumerate(indicators):
        values = []
        for key, value_list in data.items():
            if key != 'показатель' and isinstance(value_list, list):
                values.append(value_list[i])
        result[indicator] = values

    return result


# @app.route('/', methods=['GET', 'POST'])
# def index():
#     active_tab = request.args.get('tab')
#     if active_tab is None:
#         temp = get_menu_list().copy()
#         del temp['Главная страница']
#         print(os.path.join(config.DATA_PATH, 'обложка.png'),f"{config.DATA_PATH}\\обложка.png")
#         return render_template('index.html',
#                                active_tab=active_tab,
#                                tabs=temp,
#                                caption_filename=caption_filename,
#                                caption_link=caption_link)


def generate_histogram(data_matrix, group_names, axis_signatures, type="показатель"):
    groups = []
    colors = ['#ad0020', '#7d797a']
    pattern = r"^\d{4}_\d{2}_\d{2}_\d{2}_\d{2}_\d{2}$" if type == 'показатель' else r"^рейтинг$"
    group_names = [f"{item}" for item in group_names if match(pattern, item)]
    for row in range(len(data_matrix)):
        new_row = []
        for col in data_matrix[row]:
            try:
                float(col)  # Пробуем преобразовать в float, чтобы распознать целые и дробные числа
                new_row.append(col)
            except ValueError:
                pass  # Если преобразование не удалось, пропускаем элемент
        # print(type,new_row, data_matrix[row])
        data_matrix[row] = new_row[-2:] if type == "показатель" else [new_row[-3]]
        # print(type, data_matrix[row])
    if type != 'показатель':
        data_matrix = data_matrix[::-1]
    group_names = [f"{item}" for item in group_names if match(pattern, item)] if type == "показатель" else ['Рейтинг']
    for colInd, group in enumerate(group_names):
        y_val = [data_matrix[rowInd][colInd] for rowInd in range(0, len(data_matrix))]
        groups.append(
            px.Bar(name=group, x=axis_signatures,
                   y=y_val,
                   text=y_val,
                   marker_color=colors[colInd % 2]
                   )
        )
    plot = px.Figure(data=groups)
    plot.update_layout(
        barmode='group',
        plot_bgcolor='#FFFFFF',
        font=dict(size=22),
        uniformtext_minsize=12
    )
    plot.update_traces(texttemplate='%{text:.2s}', textposition='outside')

    graph = pio.to_html(plot, full_html=False, default_height=800)
    return graph


def get_list_of_params_for_query(schema: dict):
    if schema is not None:
        params = {}
        for param_name in schema.keys():
            if not (param_name == "показатель" and schema['показатель'] is None) and param_name != "имя":
                params[param_name] = []
                for baseName in inspect(db.engine).get_table_names():
                    try:
                        for item in db.session.execute(text(f"SELECT DISTINCT {param_name} FROM {baseName}")):
                            params[param_name].append(str(item)[2:-3])
                    except StatementError:
                        pass
        if len(params) > 0:
            return params
    else:
        return None


def read_dictionaries_from_file(file_path):
    """
    Читает словари из файла, разделенные абзацами, и возвращает список словарей.
    """
    dictionaries = []
    try:
        with open(file_path, 'r', encoding='utf-8') as file:
            content = file.read()
            # Разделяем контент по пустым строкам (абзацам)
            dictionary_strings = content.strip().split('\n\n')
            for dictionary_str in dictionary_strings:
                # Если строка не пустая
                if dictionary_str.strip():
                    processed_str = sub(r"(?<!\\)'", '"', dictionary_str)
                    try:
                        # Загружаем JSON строку в словарь
                        dictionary = json.loads(processed_str)
                        dictionaries.append(dictionary)
                    except json.JSONDecodeError:
                        print(f"Warning: Skipped invalid JSON string: {dictionary_str.strip()}")
    except FileNotFoundError:
        print(f"Error: File not found at path: {file_path}")
    except Exception as e:
        print(f"Error: An error occurred: {e}")
    return dictionaries


def res_to_data(query_res):
    columns = list(query_res.keys())
    dashboard_data = {column: [] for column in columns}
    rows = []
    axis_signatures = []
    query_res = query_res.fetchall()
    for row in query_res:
        rows.append(list(row))
        axis_signatures.append(row[0])
        for i, column in enumerate(columns):
            dashboard_data[column].append(row[i])
    return axis_signatures, rows, dashboard_data


def init_db():
    def clean_column_description(column_desc):
        parts = column_desc.split()
        name = "_".join(parts[:-1])

        cleaned_name = sub(r'[ /().:-]', '_', name)
        cleaned_name = cleaned_name.lower()  # lowercase for consistency

        return cleaned_name + " VARCHAR(255)"

    file_name = "Дашборд ДФУ.xlsx"
    with app.app_context():
        for table in inspect(db.engine).get_table_names():
            db.session.execute(text('DROP TABLE IF EXISTS ' + table))  # сносим старую базу

        with tqdm(total=len(pd.ExcelFile(file_name).sheet_names), desc="Сканируем базу") as pbar:
            for ind, sheet in enumerate(pd.ExcelFile(file_name).sheet_names):
                pbar.update(1)
                if sheet.lower()[:4] == "база":
                    pbar.set_description(f"{sheet} <--- база найдена!")
                    df = pd.read_excel(file_name, sheet_name=ind)  # считываем excel файл
                    num_of_cols = 0
                    sheet_name = sheet.upper() if 'СОРТИРОВКА' not in df.columns[0].upper() else \
                        df.columns[0].split("-")[1]
                    sql_create_table = f"""
                            CREATE TABLE {sheet_name} (
                            """
                    for col_name in df.columns:
                        if col_name.count(":") == 1:
                            if col_name.split(":")[0] == "Unnamed":
                                break
                        else:
                            num_of_cols += 1
                            sql_create_table += f"{clean_column_description(f'{col_name} {df[col_name].dtype.name}')}, "

                    sql_create_table = sql_create_table[:-2] + f"""
                                    );
                                    """
                    db.session.execute(text(sql_create_table))  # создаём пустую таблицу если её ещё не было
                    pbar.set_description(f"{sheet} <--- База создана!")
                    pbar.set_description(f"{sheet} <--- База заполняется...")
                    sql_insert_data = f"""INSERT INTO {sheet_name} VALUES ('"""
                    for row in df.values.tolist()[::]:
                        row = [str(val).replace("'", '"') for val in row][:num_of_cols]
                        db.session.execute(text(sql_insert_data + """', '""".join(row) + """'""" + """)"""))
                        db.session.commit()  # заполняем данными из листа "БАЗА" (регистр не важен)
                    pbar.set_description(f"{sheet} <--- База заполнена!")

        print("База данных и таблицы успешно созданы.")

    config.RELOAD_DB = False
    update_config_file('RELOAD_DB', False)


def get_menu_list():
    menu = {}
    for table_name in queries.keys():
        if (not table_name.upper().startswith("БАЗА")) or config.SHOW_BASE_TABLES:
            menu[table_name] = queries[table_name]
    return menu


def run():
    global caption_filename, queries, db, caption_link, folder_path, config
    folder_path = os.path.dirname(os.path.abspath(__file__))
    print(os.path.abspath(__file__))
    colorama.init()
    config = load_config()
    app.config['SQLALCHEMY_DATABASE_URI'] = f"sqlite:///{folder_path}/mydatabase.db"
    app.config['SQLALCHEMY_TRACK_MODIFICATIONS'] = False
    db = SQLAlchemy(app)
    log = logging.getLogger('werkzeug')
    log.disabled = not config.DEVELOPER_MODE
    host = config.HOST
    port = config.PORT
    with app.app_context():
        caption_link = read_dictionaries_from_file(os.path.join(config.DATA_PATH, f'{folder_path}/QR.txt'))[0]
        caption_filename = {}

        for key in caption_link.keys():
            generate_and_save_qr(link=caption_link[key], filename=key)

        queries = {
            "Главная страница": None,
            "Глоссарий": None
        }
        for tableName in inspect(db.engine).get_table_names():
            if tableName.lower().startswith("база"):
                queries[tableName] = f"select * from {tableName}"
        for dictionary in read_dictionaries_from_file(os.path.join(config.DATA_PATH, f'{folder_path}/tables.txt')):
            find_condition = {}
            for key in dictionary.keys():
                find_condition[key] = dictionary[key]
            TableParams(find_conditions=find_condition)

        if config.RELOAD_DB:
            init_db()
        if not config.DEVELOPER_MODE:
            webbrowser.open(f'http://{host}:{port}/table/Главная страница')
        print(
            f"{colorama.Fore.CYAN}{colorama.Style.BRIGHT} --- Закрытие консоли ведёт к прекращению работы приложения --- ")

        app.run(debug=not log.disabled,
                use_reloader=not log.disabled,
                host=host,
                port=port,
                load_dotenv=False)
